/**
 *
 * \file        CresnetConsole.c
 * \brief       This handles the redirection of the console over cresnet
 *              (abstracted from Uart_cresnet_slave)
 *
 * \author      Larry Salant
 * \date       2/6/2009
 */

////////////////////////////////////////////////////////////////////////////////

#include "Cresnetconsole.hpp"
#include "cresnet_slave.h"
//#include "uart_cresnet.h"
#include "uart_cresnet_slave.hpp"
#include "os.h"
#include <string.h>                     // for memcpy
#include "errors.h"
#include "memorymanager.h"
#include "stdlib.h"
#include "field_debug.h"

////////////////////////////////////////////////////////////////////////////////
//  Callback functions - called by pointer from the Console Task

/**
 * \author      Larry Salant
 * \date        5/27/2008
 * \brief       sends response(s) to RCON command
 * \param       pointer to data
 * \param       count - length of data
 * \param       param - contains stream address in upper word; flags in lower word
 * \return      none
 * \retval      void
 */
void CresnetConsoleRedirectTx(UINT8 *pData, UINT32 count, UINT32 param)
{
    UINT8 * pPacket;
    UINT8 bStream;
    UINT32 timeout;

    // get stream
    bStream = (UINT8)(param >> 16);

    // wait for room in the queue  and a buffer to be available
    timeout = CRESNET_QUEUE_TIMEOUT_10MS;
    while (timeout > 0)
    {
      // if there is room in the Queue 
      if (!IsCrestnetSlaveQueueFull(bStream, NET_DEST_CRESNET) )
      {
        // try to get a buffer
        // note that Buffer will be released when message is sent
        pPacket = MemMgr->GetBlock((UINT16)count+4);
        // if buffer available, continue processing; else wait and try again.
        if (pPacket)
          break;
      }

      HwDelayMsec(CRESNET_QUEUE_POLL_RATE);
      timeout--;
    }

    // get memory buffer;  
    if (timeout != 0)
    {
	  // create packet
      pPacket[0] = CNET_ID_CTRLSYS;		/* destination ID */
      
	  if (CresnetConsole->CresnetConsoleGetSource() == PASSTO_EXE)	{
		pPacket[2] = CNET_MULTISERIAL;
	  } else {
		pPacket[2] = RCON_PKT;
	  }
 	/* packet type  */
      pPacket[RCON_FLAG_IDX] = (UINT8)(param & 0xFF); // flags sent with original command
      pPacket[PACKET_LENGTH_IDX] = (UINT8)count+2;   /* packet length = number of chars + 2 for Flag and cmd */
      memcpy(&pPacket[RCON_DATA_IDX],pData,count);
      CresnetSlaveEnqueueOutgoingPacket(bStream, pPacket, count+4, NET_DEST_CRESNET);
      MemMgr->FreeBlock(pPacket);	
    }
    // else, can't report an error because we'll get into an infinite loop.
}
void CresnetConsoleRedirectCapture(void)
{
}

void CresnetConsoleRedirectRelease(void)
{
}

/**
 * \author      Larry Salant
 * \date        2/6/2009
 * \brief       constructor for Cresnet console redirection class
 * \return      none
 * \retval      void
 * \param       bNumStreams - number of streams on the card (each stream is a consecutive address)
 */
CCresnetConsole::CCresnetConsole(void)
{
    CMDINPUT * pCmdInput = &m_cmdInput;
    m_redirectActive = 0;

    pCmdInput->command.pCmdInput = pCmdInput;

    pCmdInput->pSendFunc = CresnetConsoleRedirectTx;
    pCmdInput->sendParam = 0;  // init to stream 1, will be set per message
    pCmdInput->pCaptureFunc = CresnetConsoleRedirectCapture;
    pCmdInput->pReleaseFunc = CresnetConsoleRedirectRelease;

    // init to buffer not in use
    pCmdInput->TjiPrintLock = 0;
    pCmdInput->consoleLock = 0;
    pCmdInput->printLock = 0;
    pCmdInput->captured  = 0;
	
    CmdInputAddToList(pCmdInput);
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       CresnetConsoleRedirectConnect
 * \return      none
 * \retval      void
 * \param       void
 */
void CCresnetConsole::CresnetConsoleRedirectConnect(UINT8 * packet)
{
    CMDINPUT *pCmdInput = &m_cmdInput;
  
    if ( !m_redirectActive )
    {
        m_redirectActive = 1;
		pCmdInput->cmdSource = PASSTO_EXE;
        DmConsoleDisplayWelcome(&m_cmdInput);
    }
    // ignore comspec information in message.
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       CresnetConsoleRedirectDisc
 * \return      none
 * \retval      void
 * \param       void
 */
void CCresnetConsole::CresnetConsoleRedirectDisc(UINT8 * packet)
{
    m_redirectActive = 0;
    m_cmdInput.echo = 0;
    HwDelayMsec(500);
    // trigger update request
    if (GetCresnetSlaveUart())
      GetCresnetSlaveUart()->UartCresnetSlaveSetRestartFlag(1);
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       CresnetConsoleRedirectData
 *              not called from isr
 * \return      none
 * \retval      void
 * \param       void
 */
void CCresnetConsole::CresnetConsoleRedirectData(UINT8 * packet, UINT32 packlen)
{
    UINT8 * DataPtr;
    //UINT32 flags = 0;

    // build message to send to console Task
    packlen -= 2;           // subtract off type and flags_ch bytes
    DataPtr = packet + 4;   // point to beginning of data

    while ( packlen )
    {
        // safe to ignore flags returned here since this is NOT called from
        // an ISR.
        DmConsoleBufferInput(&m_cmdInput, *DataPtr);
        DataPtr++;
        packlen--;
    }
    //return flags;
}
/**
 * \author      Larry Salant
 * \date        5/23/2008
 * \brief       sends the RCON data to the console task
 * \param       pPacket - pointer to the packet
 * \return      none
 * \retval      void
 */
BOOL CCresnetConsole::CresnetConsoleProcessRcon(UINT8 * pPacket, UINT16 sTimeout)
{
    CMDINPUT *pCmdInput = &m_cmdInput;
    UINT8 bLength;
    UINT8 *pSrc, *pDest;
    UINT32 param;

    // make sure last command has been processed since buffer is shared
    if ( !DmConsoleWaitBufferAvailable(pCmdInput, sTimeout) )
        return FALSE;

    // packet format = <id> <cnt> <1F> <00> <data>
    bLength = *(pPacket+PACKET_LENGTH_IDX) - 2;
    // make sure it fits in buffer
    if (bLength > DM_MAX_USER_CMD_LEN-1)
      bLength = DM_MAX_USER_CMD_LEN - 1;

    // copy the data into the cmd input buffer
    pDest = (UINT8 *)pCmdInput->command.commandString;
    pSrc = (UINT8 *)(pPacket+RCON_DATA_IDX);
    for (int index = 0; index < bLength; index++)
    {
      // stop if a newline or NULL is found
      if(*pSrc == EOL_CHAR || *pSrc == 0)
      {
        *pDest = 0;
        break;
      }
      *pDest = *pSrc;
      ++pDest;
      ++pSrc;
    }

    // save flag for feedback
    pCmdInput->command.length = bLength;
    // save the stream and flags from the original message
    param = GetStreamFromPacket(*pPacket) <<16;
    param |= *(UINT8 *)(pPacket+RCON_FLAG_IDX);
    pCmdInput->sendParam = param;

    // set who the message is from
    pCmdInput->cmdSource = RCON_EXE;

    // send the data portion of the message to Console task
    return ( DmConsoleSendCommandToTask(&pCmdInput->command) == -1 ? FALSE : TRUE );
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       CresnetConsoleRedirectIsActive
 * \return      none
 * \retval      void
 * \param       void
 */
BOOL CCresnetConsole::CresnetConsoleRedirectIsActive(void)
{
    return m_redirectActive;
}

/**
 * \author      Hazrat Shah
 * \date        7/15/2010
 * \brief       Get CresnetConsole source 
 * \return      UINT32 
 * \retval      Source (passto, rcon)
 * \param       void
 */
UINT32 CCresnetConsole::CresnetConsoleGetSource(void)
{
  CMDINPUT *pCmdInput = &m_cmdInput;
  
  return(pCmdInput->cmdSource);
}
